# API Documentation - Index Dar Web

## Base URL
```
http://localhost:8000/api
```

## Authentication
API ini menggunakan Laravel Sanctum untuk authentication dengan Bearer Token.

---

## Auth Endpoints

### 1. Login
**POST** `/auth/login`

**Request Body:**
```json
{
    "login": "user@example.com", // atau username
    "password": "password123"
}
```

**Success Response (200):**
```json
{
    "success": true,
    "message": "Login berhasil",
    "data": {
        "user": {
            "id": 1,
            "name": "John Doe",
            "username": "johndoe",
            "email": "john@example.com",
            "role": "Customer",
            "profile": null
        },
        "token": "1|abc123def456...",
        "token_type": "Bearer"
    }
}
```

**Error Response (401):**
```json
{
    "success": false,
    "message": "Email/Username atau password tidak valid"
}
```

**Validation Error (422):**
```json
{
    "success": false,
    "message": "Validation Error",
    "errors": {
        "login": ["Email atau Username wajib diisi"],
        "password": ["Password wajib diisi"]
    }
}
```

---

### 2. Logout
**POST** `/auth/logout`

**Headers:**
```
Authorization: Bearer {your_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "message": "Logout berhasil"
}
```

---

### 3. Get Profile
**GET** `/auth/profile`

**Headers:**
```
Authorization: Bearer {your_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "message": "Data user berhasil diambil",
    "data": {
        "user": {
            "id": 1,
            "name": "John Doe",
            "username": "johndoe",
            "email": "john@example.com",
            "role": "Customer",
            "profile": null,
            "created_at": "2025-07-31T18:30:00.000000Z",
            "updated_at": "2025-07-31T18:30:00.000000Z"
        }
    }
}
```

---

## Error Responses

### 401 Unauthorized
```json
{
    "message": "Unauthenticated."
}
```

### 500 Server Error
```json
{
    "success": false,
    "message": "Terjadi kesalahan server",
    "error": "Error details..."
}
```

---

## Usage Examples

### 1. Login dengan cURL
```bash
curl -X POST http://localhost:8000/api/auth/login \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "login": "john@example.com",
    "password": "password123"
  }'
```

### 2. Get Profile dengan cURL
```bash
curl -X GET http://localhost:8000/api/auth/profile \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Accept: application/json"
```

### 3. Logout dengan cURL
```bash
curl -X POST http://localhost:8000/api/auth/logout \
  -H "Authorization: Bearer YOUR_TOKEN_HERE" \
  -H "Accept: application/json"
```

---

## Mobile App Integration

### Android (Java/Kotlin)
```kotlin
// Login Request
val loginRequest = JSONObject()
loginRequest.put("login", "user@example.com")
loginRequest.put("password", "password123")

val request = JsonObjectRequest(
    Request.Method.POST,
    "http://your-domain.com/api/auth/login",
    loginRequest,
    { response ->
        val token = response.getJSONObject("data").getString("token")
        // Save token to SharedPreferences
    },
    { error -> 
        // Handle error
    }
)
```

### iOS (Swift)
```swift
// Login Request
let loginData = [
    "login": "user@example.com",
    "password": "password123"
]

var request = URLRequest(url: URL(string: "http://your-domain.com/api/auth/login")!)
request.httpMethod = "POST"
request.setValue("application/json", forHTTPHeaderField: "Content-Type")
request.httpBody = try JSONSerialization.data(withJSONObject: loginData)
```

### Flutter/Dart
```dart
// Login Request
final response = await http.post(
  Uri.parse('http://your-domain.com/api/auth/login'),
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
  body: jsonEncode({
    'login': 'user@example.com',
    'password': 'password123',
  }),
);

if (response.statusCode == 200) {
  final data = jsonDecode(response.body);
  final token = data['data']['token'];
  // Save token for future requests
}
```

---

## Notes

1. **Token Storage**: Simpan token di secure storage (SharedPreferences di Android, Keychain di iOS)
2. **Token Expiration**: Token tidak memiliki expiration time secara default, tapi bisa di-logout secara manual
3. **Headers**: Selalu sertakan `Accept: application/json` untuk mendapatkan response dalam format JSON
4. **HTTPS**: Gunakan HTTPS di production untuk keamanan
