import mysql.connector
from mysql.connector import Error
import os
from dotenv import load_dotenv
from passlib.context import CryptContext

load_dotenv()
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

def create_connection():
    """Membuat koneksi ke database MySQL"""
    try:
        connection = mysql.connector.connect(
            host=os.getenv("DB_HOST"),
            user=os.getenv("DB_USER"),
            password=os.getenv("DB_PASSWORD"),
            database=os.getenv("DB_NAME")
        )
        return connection
    except Error as e:
        print(f"Database connection error: {e}")
        raise

def init_db():
    """Inisialisasi database dan tabel"""
    connection = None
    cursor = None
    try:
        connection = create_connection()
        cursor = connection.cursor()
        
        # Tabel Pengguna
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS pengguna (
            id INT AUTO_INCREMENT PRIMARY KEY,
            username VARCHAR(50) UNIQUE NOT NULL,
            email VARCHAR(100) UNIQUE NOT NULL,
            password_hash VARCHAR(255) NOT NULL,
            nama_lengkap VARCHAR(100),
            role ENUM('admin', 'guru', 'siswa') NOT NULL,
            tanggal_dibuat DATETIME DEFAULT CURRENT_TIMESTAMP,
            terakhir_login DATETIME
        )
        """)
        
        # Tabel Tingkat (SD, SMP, SMA)
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS tingkat (
            id INT AUTO_INCREMENT PRIMARY KEY,
            nama VARCHAR(50) NOT NULL
        )
        """)

        # Tabel Kelas
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS kelas (
            id INT AUTO_INCREMENT PRIMARY KEY,
            tingkat_id INT NOT NULL,
            nama_kelas VARCHAR(100) NOT NULL,
            FOREIGN KEY (tingkat_id) REFERENCES tingkat(id)
        )
        """)
        
        # Tabel Pelajaran
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS pelajaran (
            id INT AUTO_INCREMENT PRIMARY KEY,
            tingkat_id INT NOT NULL,
            nama VARCHAR(100) NOT NULL,
            deskripsi TEXT,
            gambar_url VARCHAR(255),
            FOREIGN KEY (tingkat_id) REFERENCES tingkat(id)
        )
        """)
        
        # Tambahkan kolom kelas_id ke tabel pelajaran jika belum ada
        cursor.execute("""
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_NAME = 'pelajaran' AND COLUMN_NAME = 'kelas_id'
        """)
        kelas_id_exists = cursor.fetchone()

        if not kelas_id_exists:
            cursor.execute("""
            ALTER TABLE pelajaran
            ADD COLUMN kelas_id INT AFTER tingkat_id,
            ADD CONSTRAINT fk_kelas_id FOREIGN KEY (kelas_id) REFERENCES kelas(id)
            """)
        
        # Tabel Materi
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS materi (
            id INT AUTO_INCREMENT PRIMARY KEY,
            pelajaran_id INT NOT NULL,
            judul VARCHAR(255) NOT NULL,
            konten TEXT,
            tipe ENUM('teks', 'video', 'pdf', 'gambar') NOT NULL,
            urutan INT NOT NULL,
            dibuat_oleh INT NOT NULL,
            tanggal_dibuat DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (pelajaran_id) REFERENCES pelajaran(id),
            FOREIGN KEY (dibuat_oleh) REFERENCES pengguna(id)
        )
        """)

        # Tambahkan kolom file_path dan file_url jika belum ada
        cursor.execute("""
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_NAME = 'materi' AND COLUMN_NAME = 'file_path'
        """)
        file_path_exists = cursor.fetchone()

        if not file_path_exists:
            cursor.execute("""
            ALTER TABLE materi
            ADD COLUMN file_path VARCHAR(255) AFTER urutan
            """)

        cursor.execute("""
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_NAME = 'materi' AND COLUMN_NAME = 'file_url'
        """)
        file_url_exists = cursor.fetchone()

        if not file_url_exists:
            cursor.execute("""
            ALTER TABLE materi
            ADD COLUMN file_url VARCHAR(255) AFTER file_path
            """)
        
        # Tabel Soal
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS soal (
            id INT AUTO_INCREMENT PRIMARY KEY,
            pelajaran_id INT NOT NULL,
            materi_id INT,
            pertanyaan TEXT NOT NULL,
            pilihan_a VARCHAR(255) NOT NULL,
            pilihan_b VARCHAR(255) NOT NULL,
            pilihan_c VARCHAR(255) NOT NULL,
            pilihan_d VARCHAR(255) NOT NULL,
            jawaban_benar ENUM('a', 'b', 'c', 'd') NOT NULL,
            tingkat_kesulitan ENUM('mudah', 'sedang', 'sulit') DEFAULT 'sedang',
            dibuat_oleh INT NOT NULL,
            FOREIGN KEY (pelajaran_id) REFERENCES pelajaran(id),
            FOREIGN KEY (materi_id) REFERENCES materi(id),
            FOREIGN KEY (dibuat_oleh) REFERENCES pengguna(id)
        )
        """)
        
        # Hapus foreign key materi_id jika ada
        cursor.execute("""
        SELECT CONSTRAINT_NAME 
        FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
        WHERE TABLE_NAME = 'soal' AND COLUMN_NAME = 'materi_id';
        """)
        fk_name = cursor.fetchone()

        if fk_name:
            cursor.execute(f"ALTER TABLE soal DROP FOREIGN KEY {fk_name[0]}")
        
        # Tabel Feedback Materi
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS feedback_materi (
            id INT AUTO_INCREMENT PRIMARY KEY,
            materi_id INT NOT NULL,
            pengguna_id INT NOT NULL,
            rating TINYINT NOT NULL,
            komentar TEXT,
            tanggal_dibuat DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (materi_id) REFERENCES materi(id),
            FOREIGN KEY (pengguna_id) REFERENCES pengguna(id)
        )
        """)
        
        # Insert data dasar jika tabel kosong
        cursor.execute("SELECT COUNT(*) FROM pengguna")
        if cursor.fetchone()[0] == 0:
            # Buat admin default
            password_hash = pwd_context.hash("admin123")
            cursor.execute("""
            INSERT INTO pengguna (username, email, password_hash, nama_lengkap, role)
            VALUES (%s, %s, %s, %s, %s)
            """, ("admin", "admin@example.com", password_hash, "Administrator", "admin"))
            
            # Insert tingkat pendidikan
            cursor.execute("INSERT INTO tingkat (nama) VALUES ('SD'), ('SMP'), ('SMA')")
            
        # Tabel Hasil Ujian
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS hasil_ujian (
            id INT AUTO_INCREMENT PRIMARY KEY,
            pelajaran_id INT NOT NULL,
            siswa_id INT NOT NULL,
            tanggal_mulai DATETIME NOT NULL,
            tanggal_selesai DATETIME,
            nilai DECIMAL(5,2),
            status ENUM('dalam_proses', 'selesai', 'terlewat') DEFAULT 'dalam_proses',
            FOREIGN KEY (pelajaran_id) REFERENCES pelajaran(id),
            FOREIGN KEY (siswa_id) REFERENCES pengguna(id)
        )
        """)

        # Tabel Jawaban Siswa
        cursor.execute("""
        CREATE TABLE IF NOT EXISTS jawaban_siswa (
            hasil_ujian_id INT NOT NULL,
            soal_id INT NOT NULL,
            jawaban_dipilih ENUM('a', 'b', 'c', 'd'),
            is_benar BOOLEAN,
            PRIMARY KEY (hasil_ujian_id, soal_id),
            FOREIGN KEY (hasil_ujian_id) REFERENCES hasil_ujian(id),
            FOREIGN KEY (soal_id) REFERENCES soal(id)
        )
        """)
        
            
        connection.commit()
    except Error as e:
        print(f"Database error: {e}")
        if connection:
            connection.rollback()
        raise
    finally:
        if cursor:
            cursor.close()
        if connection and connection.is_connected():
            connection.close()