from flask import Flask, render_template, request, redirect, url_for, session, flash, send_file, jsonify
from generate_pdf import generate_pdf, generate_pdf_bytes
from functools import wraps
from werkzeug.security import generate_password_hash, check_password_hash
import os
import json
import base64
import datetime

app = Flask(__name__)
app.secret_key = "supersecretkey"

USER_FILE = "users.json"
LOG_FILE = "logs.json"

# ==================== UTILITY ====================
def load_users():
    if not os.path.exists(USER_FILE):
        return {}
    with open(USER_FILE, "r") as f:
        return json.load(f)

def save_users(users):
    with open(USER_FILE, "w") as f:
        json.dump(users, f, indent=2)

def append_log(username, template_name):
    log = {
        "username": username,
        "template": template_name,
        "timestamp": datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    }
    if os.path.exists(LOG_FILE):
        with open(LOG_FILE, "r") as f:
            logs = json.load(f)
    else:
        logs = []
    logs.append(log)
    with open(LOG_FILE, "w") as f:
        json.dump(logs, f, indent=2)

def load_config(template_name):
    config_path = f"template_configs/{template_name}.json"
    with open(config_path, "r") as f:
        return json.load(f)

# ==================== LOGIN PROTECT ====================
def login_required(f):
    @wraps(f)
    def wrapper(*args, **kwargs):
        if "user" not in session:
            flash("Silakan login terlebih dahulu.")
            return redirect(url_for("login"))
        return f(*args, **kwargs)
    return wrapper

# ==================== ROUTES ====================
@app.route("/", methods=["GET", "POST"])
@login_required
def index():
    template_list = [f[:-5] for f in os.listdir("template_configs") if f.endswith(".json")]
    selected_template = request.form.get("template") if request.method == "POST" else template_list[0]
    config = load_config(selected_template)

    if request.method == "POST":
        form_data = request.form.to_dict()
        output_path = os.path.join("static", "preview.pdf")
        generate_pdf(form_data, output_path, template_name=selected_template)
        append_log(session["user"], selected_template)
        return render_template("form.html", preview=True, pdf_path=output_path, config=config,
                               templates=template_list, selected_template=selected_template,
                               username=session["user"])

    return render_template("form.html", preview=False, config=config,
                           templates=template_list, selected_template=selected_template,
                           username=session["user"])

@app.route("/preview-pdf-bytes", methods=["POST"])
@login_required
def preview_pdf_bytes():
    form_data = request.form.to_dict()
    template_name = form_data.get("template")
    pdf_bytes = generate_pdf_bytes(form_data, template_name)
    encoded = base64.b64encode(pdf_bytes).decode("utf-8")
    return jsonify({"pdf_base64": encoded})

@app.route("/download")
@login_required
def download():
    path = os.path.join("static", "preview.pdf")
    return send_file(path, as_attachment=True)

@app.route("/login", methods=["GET", "POST"])
def login():
    users = load_users()
    if request.method == "POST":
        username = request.form["username"]
        password = request.form["password"]

        if username in users and check_password_hash(users[username]["password"], password):
            session["user"] = username
            return redirect(url_for("index"))
        else:
            flash("Username atau password salah!")
    return render_template("login.html")

@app.route("/register", methods=["GET", "POST"])
def register():
    users = load_users()
    if request.method == "POST":
        username = request.form["username"]
        password = request.form["password"]

        if username in users:
            flash("Username sudah digunakan.")
        else:
            users[username] = {
                "password": generate_password_hash(password),
                "role": "user"
            }
            save_users(users)
            flash("Registrasi berhasil, silakan login.")
            return redirect(url_for("login"))
    return render_template("register.html")

@app.route("/logout")
def logout():
    session.pop("user", None)
    flash("Berhasil logout.")
    return redirect(url_for("login"))

@app.route("/admin")
@login_required
def admin():
    users = load_users()
    current_user = session["user"]

    if users.get(current_user, {}).get("role") != "admin":
        flash("Akses ditolak. Admin only.")
        return redirect(url_for("index"))

    if os.path.exists(LOG_FILE):
        with open(LOG_FILE, "r") as f:
            logs = json.load(f)
    else:
        logs = []

    return render_template("admin.html", logs=logs, users=users)

# ==================== RUN ====================
if __name__ == "__main__":
    app.run(debug=True)
