<?php

namespace App\Http\Controllers;

use App\Models\CartItem;
use App\Models\Inventory;
use App\Models\Pelacakan;
use App\Models\Pelanggan;
use App\Models\Product;
//cicilan
use App\Models\CicilanHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class HomepageController extends Controller
{
    public function index()
    {
        $products = Product::where('best_seller', 0)
            ->where('stok_produk', '>', 0)
            ->latest()
            ->paginate(5);
        $products_best = Product::where('best_seller', 1)
            ->where('stok_produk', '>', 0)
            ->latest()
            ->paginate(4);
        $cartItems = CartItem::with('produk')->where('id_user', Auth::id())->get();
        $cartCount = $cartItems->sum('quantity');
        $totalPrice = $cartItems->sum(function ($item) {
            return $item->quantity * $item->produk->harga_produk;
        });

        return view('pages.homepage.index', compact('products', 'products_best', 'cartItems', 'cartCount', 'totalPrice'));
    }

    public function getProductById(string $id)
    {
        $product = Product::findOrFail($id);
        $cartItems = CartItem::with('produk')->where('id_user', Auth::id())->get();
        $cartCount = $cartItems->sum('quantity');
        $totalPrice = $cartItems->sum(function ($item) {
            return $item->quantity * $item->produk->harga_produk;
        });

        return view('pages.homepage.product', compact('product', 'cartItems', 'cartCount', 'totalPrice'));
    }

    public function search(Request $request)
    {
        $search = $request->input('search');
        $products = Product::where('nama_produk', 'like', '%' . $search . '%')->where('stok_produk', '>', 0)->latest()->paginate(5);
        if ($search == null) {
            $products = Product::where('best_seller', 0)
                ->where('stok_produk', '>', 0)
                ->latest()
                ->paginate(5);
            $products_best = Product::where('best_seller', 1)->where('stok_produk', '>', 0)->latest()->paginate(4);
        } else {
            $products_best = null;
        }

        return view('pages.homepage.index', compact('products', 'products_best'));
    }

    public function getCart()
    {
        $cartItems = CartItem::with('produk')->where('id_user', Auth::id())->get();
        $cartCount = $cartItems->sum('quantity');
        $totalPrice = $cartItems->sum(function ($item) {
            return $item->quantity * $item->produk->harga_produk;
        });

        return view('pages.homepage.cart', compact('cartItems', 'cartCount', 'totalPrice'));
    }

    public function getCartDataResponse($message = '')
    {
        $userId = Auth::id();
        $cartItems = CartItem::with('produk')
            ->where('id_user', $userId)
            ->get();

        $totalPrice = $cartItems->sum(function ($item) {
            return $item->produk->harga_produk * $item->quantity;
        });

        $cartCount = CartItem::where('id_user', $userId)->sum('quantity');

        $items = $cartItems->map(function ($item) {
            return [
                'id' => $item->produk->id,
                'nama' => $item->produk->nama_produk,
                'foto' => asset('storage/foto_produk/' . $item->produk->foto_produk),
                'harga' => number_format($item->produk->harga_produk, 0, ',', '.'),
                'quantity' => $item->quantity,
                'remove_url' => route('product.remove.cart', $item->produk->id),
                'detail_url' => route('product.detail', $item->produk->id),
            ];
        });

        return response()->json([
            'message' => $message,
            'cartItems' => $items,
            'cartCount' => $cartCount,
            'total' => number_format($totalPrice, 0, ',', '.'),
            'checkout_url' => route('product.checkout'),
        ]);
    }

    public function addToCart_v1(Request $request, $id)
    {
        $product = Product::findOrFail($id);
        $userId = Auth::id();
        $quantity = $request->input('quantity', 1);

        $cartItem = CartItem::where('id_user', $userId)
            ->where('id_produk', $id)
            ->first();

        if ($cartItem) {
            $cartItem->quantity += $quantity;
            $cartItem->save();
        } else {
            CartItem::create([
                'id_user' => $userId,
                'id_produk' => $id,
                'quantity' => $quantity,
            ]);
        }

        return $this->getCartDataResponse('Produk berhasil ditambahkan ke keranjang');
    }

    public function addToCart(Request $request, $id)
    {
        $user = Auth::user();

        // Cek apakah user memiliki utang
        if ($user->ada_hutang == 1) {
            return response()->json([
                'message' => 'Lunasi pesanan sebelumnya terlebih dahulu untuk melanjutkan pembelian!',
            ], 403);
        }

        $product = Product::findOrFail($id);
        $userId = $user->id;
        $quantity = $request->input('quantity', 1);

        $cartItem = CartItem::where('id_user', $userId)
            ->where('id_produk', $id)
            ->first();

        if ($cartItem) {
            $cartItem->quantity += $quantity;
            $cartItem->save();
        } else {
            CartItem::create([
                'id_user' => $userId,
                'id_produk' => $id,
                'quantity' => $quantity,
            ]);
        }

        return $this->getCartDataResponse('Produk berhasil ditambahkan ke keranjang');
    }


    public function updateCart(Request $request, $id)
    {
        $item = CartItem::where('id_user', auth()->id())->where('id_produk', $id)->firstOrFail();
        $item->quantity = $request->input('quantity');
        $item->save();

        return response()->json(['message' => 'Jumlah produk diperbarui.']);
    }

    public function removeFromCart($id)
    {
        $userId = Auth::id();

        CartItem::where('id_user', $userId)
            ->where('id_produk', $id)
            ->delete();

        $cartCount = CartItem::where('id_user', $userId)->sum('quantity');
        $totalPrice = CartItem::where('id_user', $userId)
            ->join('products', 'cart_items.id_produk', '=', 'products.id')
            ->sum(DB::raw('cart_items.quantity * products.harga_produk'));

        return response()->json([
            'message' => 'Produk berhasil dihapus dari keranjang',
            'cartCount' => $cartCount,
            'totalPrice' => $totalPrice,
        ]);
    }

    public function getCheckout11()
    {
        $userId = Auth::id();
        $cartItems = CartItem::with('produk')->where('id_user', $userId)->get();
        $cartCount = CartItem::where('id_user', $userId)->sum('quantity');
        $totalPrice = $cartItems->sum(function ($item) {
            return $item->quantity * $item->produk->harga_produk;
        });

        return view('pages.homepage.checkout', compact('cartItems', 'cartCount', 'totalPrice'));
    }

    public function getCheckout()
    {
        $userId = Auth::id();
        $user = Auth::user();

        if ($user->ada_hutang == 1) {
            return redirect()->back()->withErrors([
                'error' => 'Lunasi pesanan sebelumnya terlebih dahulu sebelum melakukan checkout.'
            ]);
        }

        $cartItems = CartItem::with('produk')->where('id_user', $userId)->get();
        $cartCount = CartItem::where('id_user', $userId)->sum('quantity');
        $totalPrice = $cartItems->sum(function ($item) {
            return $item->quantity * $item->produk->harga_produk;
        });

        return view('pages.homepage.checkout', compact('cartItems', 'cartCount', 'totalPrice'));
    }

    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'password' => 'nullable|min:6|confirmed',
            'phone' => ['nullable', 'regex:/^[0-9]{1,16}$/'],
            'alamat_pelanggan' => 'nullable|string|max:255',
        ]);

        try {
            DB::transaction(function () use ($request, $user) {
                // Update data di tabel users
                $user->name = $request->name;
                $user->email = $request->email;

                if ($request->filled('password')) {
                    $user->password = Hash::make($request->password);
                }

                $user->save();

                // Update data di tabel pelanggans (berdasarkan user_id)
                Pelanggan::where('user_id', $user->id)->update([
                    'nama_pelanggan' => $request->name,
                    'phone' => $request->phone,
                    'alamat_pelanggan' => $request->alamat_pelanggan,
                ]);
            });

            return redirect()->back()->with('success', 'Profil berhasil diperbarui.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Gagal memperbarui profil: ' . $e->getMessage()]);
        }
    }

    public function getOrder(Request $request)
    {
        $user = Auth::user();
        $pelanggan = $user->pelanggan;

        if (!$pelanggan) {
            return redirect()->back()->withErrors(['error' => 'Data pelanggan tidak ditemukan.']);
        }

        $query = Pelacakan::with(['products', 'kurir', 'inventory'])
            ->where('id_customer', $pelanggan->id);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $orders = $query->orderBy('created_at', 'desc')->get();

        return view('pages.homepage.order', compact('orders'));
    }

    public function order(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => ['required', 'regex:/^[0-9]{1,16}$/'],
            'alamat_pelanggan' => 'required|string|max:255',
            'metode_bayar' => 'nullable|in:lunas,cicilan',
            'periode_cicilan' => 'nullable|in:3,6,12',
            'bukti' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $cartItems = CartItem::where('id_user', $user->id)->with('produk')->get();

        if ($cartItems->isEmpty()) {
            return redirect()->back()->withErrors(['error' => 'Keranjang kosong.']);
        }

        try {
            DB::transaction(function () use ($request, $cartItems, $user) {
                // Upload bukti pembayaran jika ada
                $bukti = null;
                if ($request->hasFile('bukti')) {
                    $buktiPath = $request->file('bukti')->store('bukti_pembayaran', 'public');
                    $bukti = str_replace('bukti_pembayaran/', '', $buktiPath);
                }

                // Update user info
                $user->name = $request->name;
                $user->email = $request->email;
                $user->save();

                // Update pelanggan info
                $pelanggan = Pelanggan::where('user_id', $user->id)->firstOrFail();
                $pelanggan->nama_pelanggan = $request->name;
                $pelanggan->phone = $request->phone;
                $pelanggan->alamat_pelanggan = $request->alamat_pelanggan;
                $pelanggan->save();

                // Hitung total harga semua item
                $total_harga = $cartItems->sum(function ($item) {
                    return $item->produk->harga_produk * $item->quantity;
                });

                // Tentukan metode bayar & hitung cicilan jika dipilih
                $cara_bayar = $request->metode_bayar ?? 'lunas';
                $periodeCicilan = null;
                $jumlah_pelunasan = $total_harga;
                $sisa_pelunasan = 0;

                if ($cara_bayar === 'cicilan') {
                    $periodeCicilan = $request->periode_cicilan;
                    $jumlah_pelunasan = ceil($total_harga / $periodeCicilan); // cicilan bulan pertama
                    $sisa_pelunasan = $total_harga - $jumlah_pelunasan;

                    // update user ada_hutang
                    $user->ada_hutang = 1;
                    $user->save();
                }

                // Simpan pelacakan (order)
                $pelacakan = Pelacakan::create([
                    'status' => 'dikemas',
                    'id_customer' => $pelanggan->id,
                    'total' => $total_harga,
                    'jumlah_pelunasan' => $jumlah_pelunasan,
                    'sisa_pelunasan' => $sisa_pelunasan,
                    'bukti' => $bukti,
                    'cara_bayar' => $cara_bayar,
                    'periode_cicilan' => $periodeCicilan,
                ]);

                // Proses produk dalam keranjang
                foreach ($cartItems as $item) {
                    $product = Product::where('id', $item->id_produk)->lockForUpdate()->first();

                    if ($item->quantity > $product->stok_produk) {
                        throw new \Exception("Stok produk '{$product->nama_produk}' tidak mencukupi.");
                    }

                    $product->stok_produk -= $item->quantity;
                    $product->save();

                    // Simpan ke pivot table order_details
                    $pelacakan->products()->attach($item->id_produk, [
                        'harga_produk' => $item->produk->harga_produk,
                        'quantity' => $item->quantity,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);

                    // Tambah ke inventory
                    Inventory::create([
                        'jenis' => 'barang keluar',
                        'jumlah_barang' => $item->quantity,
                        'id_produk' => $product->id,
                        'id_karyawan' => null,
                        'pesan' => 'Pembelian ' . $product->nama_produk . ' oleh ' . $pelanggan->nama_pelanggan,
                        'pembayaran' => $item->produk->harga_produk * $item->quantity,
                        'id_pelacakan' => $pelacakan->id,
                    ]);
                }

                // Hapus isi keranjang
                CartItem::where('id_user', $user->id)->delete();
            });

            return redirect()->route('product.order')->with('success', 'Pesanan berhasil dibuat.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Gagal membuat pesanan: ' . $e->getMessage()]);
        }
    }


    public function bayarCicilan(Request $request, $id)
    {
        $order = Pelacakan::findOrFail($id);

        // Pastikan order milik user yang login
        $pelanggan = Auth::user()->pelanggan;
        if (!$pelanggan || $order->id_customer != $pelanggan->id) {
            abort(403);
        }

        $request->validate([
            'metode_pembayaran' => 'required|in:transfer,tunai',
            'bukti' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        try {
            DB::transaction(function () use ($request, $order, $pelanggan) {
                // Upload bukti jika transfer
                $bukti = null;
                if ($request->metode_pembayaran === 'transfer' && $request->hasFile('bukti')) {
                    $buktiPath = $request->file('bukti')->store('bukti_pembayaran', 'public');
                    $bukti = str_replace('bukti_pembayaran/', '', $buktiPath);
                }

                // Hitung jumlah cicilan otomatis
                $jumlah_cicilan = ceil($order->total / max(1, $order->periode_cicilan));
                $jumlah_cicilan = min($jumlah_cicilan, $order->sisa_pelunasan); // Jangan melebihi sisa

                // Simpan riwayat cicilan
                CicilanHistory::create([
                    'pelacakan_id' => $order->id,
                    'pelanggan_id' => $pelanggan->id,
                    'jumlah_cicilan' => $jumlah_cicilan,
                    'metode_pembayaran' => $request->metode_pembayaran,
                    'bukti' => $bukti,
                ]);

                // Update pelacakan
                $order->jumlah_pelunasan += $jumlah_cicilan;
                $order->sisa_pelunasan = max(0, $order->total - $order->jumlah_pelunasan);

                // Jika sudah lunas, update status dan hutang user
                if ($order->sisa_pelunasan <= 0) {
                    $order->sisa_pelunasan = 0;
                    // $order->status = 'selesai'; // ✅ opsional
                    $user = $pelanggan->user;
                    if ($user) {
                        $user->ada_hutang = 0;
                        $user->save();
                    }
                }

                // Simpan bukti jika ada
                if ($bukti) {
                    $order->bukti = $bukti;
                }

                $order->save();
            });

            return redirect()->route('product.order')->with('success', 'Pembayaran cicilan berhasil.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Gagal membayar cicilan: ' . $e->getMessage()]);
        }
    }




    public function formBayarCicilan($id)
    {
        $order = Pelacakan::findOrFail($id);

        // Pastikan order milik user yang login
        $pelanggan = Auth::user()->pelanggan;
        if (!$pelanggan || $order->id_customer != $pelanggan->id) {
            abort(403);
        }

        $histories = CicilanHistory::where('pelacakan_id', $order->id)->orderBy('created_at')->get();

        return view('pages.homepage.bayar-cicilan', compact('order', 'histories'));
    }
}
