<?php

namespace App\Http\Controllers;

use App\Models\Klasifikasi;
use App\Models\DetailKlasifikasi;
use App\Models\Gatra;
use Illuminate\Http\Request;

class KlasifikasiController extends Controller
{
    public function index()
    {
        $no = 1;
        $data_klasifikasi = Klasifikasi::orderBy('gatra')->get();
        return view('pages.data-klasifikasi.index', compact('no', 'data_klasifikasi'));
    }

    public function create()
    {
        return view('pages.klasifikasi.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'gatra' => 'required|string|max:255',
            'type' => 'required|string|max:255',
            'turvar_id' => 'nullable|integer',
            'tahun_id' => 'nullable|integer',
            'turtahun_id' => 'nullable|integer',
            'jenis_tangguh' => 'required|string|max:255',
            'batas_bawah' => 'required|numeric',
            'batas_atas' => 'required|numeric',
            'api' => 'required|string|max:255',
            'json_data' => 'required|string',
        ]);

        // Parse JSON data
        $jsonData = json_decode($validated['json_data'], true);
        
        if (!$jsonData) {
            return back()->withErrors(['json_data' => 'Format JSON tidak valid'])->withInput();
        }

        // Extract data from JSON
        $subject = $jsonData['subject'][0] ?? null;
        $var = $jsonData['var'][0] ?? null;
        $turvar = null;
        $tahun = null;
        $turtahun = null;

        if (!$subject || !$var) {
            return back()->withErrors(['json_data' => 'Data subject atau var tidak ditemukan dalam JSON'])->withInput();
        }

        // Get turvar data from JSON if turvar_id is provided
        if ($validated['turvar_id'] && $validated['turvar_id'] != 0) {
            $turvarList = $jsonData['turvar'] ?? [];
            foreach ($turvarList as $turvarItem) {
                if ($turvarItem['val'] == $validated['turvar_id']) {
                    $turvar = $turvarItem['label'];
                    break;
                }
            }
        } else {
            // If turvar_id is 0 or empty, check if turvar array has "Tidak ada" entry
            $turvarList = $jsonData['turvar'] ?? [];
            if (!empty($turvarList)) {
                $firstTurvar = $turvarList[0];
                if ($firstTurvar['val'] == 0 || $firstTurvar['val'] == "0") {
                    $turvar = $firstTurvar['label'];
                }
            }
            if (!$turvar) {
                $turvar = 'Tidak ada';
            }
        }

        // Get tahun data from JSON if tahun_id is provided
        if ($validated['tahun_id']) {
            $tahunList = $jsonData['tahun'] ?? [];
            foreach ($tahunList as $tahunItem) {
                if ($tahunItem['val'] == $validated['tahun_id']) {
                    $tahun = $tahunItem['label'];
                    break;
                }
            }
        }

        // Get turtahun data from JSON if turtahun_id is provided
        if ($validated['turtahun_id'] && $validated['turtahun_id'] != 0) {
            $turtahunList = $jsonData['turtahun'] ?? [];
            foreach ($turtahunList as $turtahunItem) {
                if ($turtahunItem['val'] == $validated['turtahun_id']) {
                    $turtahun = $turtahunItem['label'];
                    break;
                }
            }
        } else {
            // If turtahun_id is 0 or empty, check if turtahun array has default entry
            $turtahunList = $jsonData['turtahun'] ?? [];
            if (!empty($turtahunList)) {
                $firstTurtahun = $turtahunList[0];
                if ($firstTurtahun['val'] == 0 || $firstTurtahun['val'] == "0") {
                    $turtahun = $firstTurtahun['label'];
                }
            }
            if (!$turtahun) {
                $turtahun = 'Tahun';
            }
        }

        // Calculate step values
        $batasBawah = $validated['batas_bawah'];
        $batasAtas = $validated['batas_atas'];
        $step = ($batasAtas - $batasBawah) / 5;

        $data = [
            'gatra' => $validated['gatra'],
            'type' => $validated['type'],
            'subject_id' => $subject['val'],
            'subject' => $subject['label'],
            'var_id' => $var['val'],
            'var' => $var['label'],
            'turvar_id' => $validated['turvar_id'] ?? 0,
            'turvar' => $turvar,
            'tahun_id' => $validated['tahun_id'] ?? 0,
            'tahun' => $tahun,
            'turtahun_id' => $validated['turtahun_id'] ?? 0,
            'turtahun' => $turtahun,
            'jenis_tangguh' => $validated['jenis_tangguh'],
            'batas_bawah' => $batasBawah,
            'batas_atas' => $batasAtas,
            'batas_atas1' => $batasBawah + ($step * 1),
            'batas_atas2' => $batasBawah + ($step * 2),
            'batas_atas3' => $batasBawah + ($step * 3),
            'batas_atas4' => $batasBawah + ($step * 4),
            'batas_atas5' => $batasBawah + ($step * 5),
            'title' => $var['label'],
            'api' => $validated['api'],
            'kode' => null,
            'indeks' => 0,
        ];

        Klasifikasi::create($data);

        // Get the created klasifikasi ID
        $klasifikasiId = Klasifikasi::latest()->first()->id;

        // Parse and insert detail_klasifikasi data
        $this->parseAndInsertDetailKlasifikasi($jsonData, $klasifikasiId, $data);

        return redirect()->route('klasifikasi.index')->with('success', 'Data klasifikasi berhasil ditambahkan.');
    }

    private function parseAndInsertDetailKlasifikasi($jsonData, $klasifikasiId, $parentData)
    {
        $datacontent = $jsonData['datacontent'] ?? [];
        $vervarList = $jsonData['vervar'] ?? [];
        $labelvervar = $jsonData['labelvervar'] ?? '';

        // Process all datacontent entries
        foreach ($datacontent as $datacontentId => $datacontentVal) {
            // Parse datacontent_id to get components with vervar validation
            $components = $this->parseDatacontentId($datacontentId, $vervarList, $parentData['var_id'], $parentData['tahun_id']);
            
            // Find vervar data based on parsed vervar_id
            $vervarData = collect($vervarList)->firstWhere('val', (int)$components['vervar_id']);
            
            if ($vervarData && $datacontentVal > 0) {
                // Calculate nilai_indeks based on datacontent_val and klasifikasi boundaries
                $nilaiIndeks = $this->calculateNilaiIndeks($datacontentVal, $parentData);

                // Handle turvar for detail record
                $detailTurvar = $parentData['turvar'];
                $detailTurvarId = $parentData['turvar_id'];
                
                // Handle turtahun for detail record  
                $detailTurtahun = $parentData['turtahun'];
                $detailTurtahunId = $parentData['turtahun_id'];

                DetailKlasifikasi::create([
                    'klasifikasi_id' => $klasifikasiId,
                    'gatra' => $parentData['gatra'],
                    'subject_id' => $parentData['subject_id'],
                    'subject' => $parentData['subject'],
                    'turvar_id' => $detailTurvarId,
                    'turvar' => $detailTurvar,
                    'var_id' => $parentData['var_id'],
                    'var' => $parentData['var'],
                    'vervar_id' => $vervarData['val'],
                    'labelvervar' => $labelvervar,
                    'vervar' => $vervarData['label'],
                    'prov_id' => $vervarData['val'], // prov_id sama dengan vervar_id
                    'prov' => $vervarData['label'], // prov sama dengan vervar
                    'tahun_id' => $parentData['tahun_id'],
                    'tahun' => $parentData['tahun'],
                    'turtahun_id' => $detailTurtahunId,
                    'turtahun' => $detailTurtahun,
                    'datacontent_id' => $datacontentId,
                    'datacontent_val' => (float)$datacontentVal,
                    'nilai_indeks' => $nilaiIndeks,
                ]);
            }
        }
    }

    private function parseDatacontentId($datacontentId, $vervarList = [], $varId = null, $tahunId = null)
    {
        // Extract components from datacontent_id with better logic
        // We need to match against known vervar values to determine the correct parsing
        
        $length = strlen($datacontentId);
        
        // Get all possible vervar values for matching
        $vervarValues = collect($vervarList)->pluck('val')->map(function($val) {
            return (int)$val;
        })->sort()->reverse(); // Sort descending to try longer matches first
        
        // Try to find the correct vervar_id by matching against known values
        foreach ($vervarValues as $vervarVal) {
            $vervarStr = (string)$vervarVal;
            $vervarLength = strlen($vervarStr);
            
            // Check if datacontent_id starts with this vervar value
            if (substr($datacontentId, 0, $vervarLength) == $vervarStr) {
                $remainingId = substr($datacontentId, $vervarLength);
                $remainingLength = strlen($remainingId);
                
                // For the pattern: var(3) + turvar(1) + tahun(3) + turtahun(1) = 8 chars remaining
                if ($remainingLength >= 8) {
                    return [
                        'vervar_id' => (int)$vervarVal,
                        'var_id' => (int)substr($remainingId, 0, 3),      // Next 3 digits
                        'turvar_id' => (int)substr($remainingId, 3, 1),   // Next 1 digit
                        'tahun_id' => (int)substr($remainingId, 4, 3),    // Next 3 digits
                        'turtahun_id' => (int)substr($remainingId, 7, 1), // Last 1 digit
                    ];
                }
                // For other patterns, try different combinations
                elseif ($remainingLength >= 7) {
                    return [
                        'vervar_id' => (int)$vervarVal,
                        'var_id' => (int)substr($remainingId, 0, 3),      
                        'turvar_id' => (int)substr($remainingId, 3, 1),   
                        'tahun_id' => (int)substr($remainingId, 4, 3),    
                        'turtahun_id' => 0, // Default if not enough chars
                    ];
                }
            }
        }
        
        // Fallback to original logic if no match found
        if ($length == 10) {
            // Format: vervar(2) + var(3) + turvar(1) + tahun(3) + turtahun(1) = 10 chars
            return [
                'vervar_id' => (int)substr($datacontentId, 0, 2),   
                'var_id' => (int)substr($datacontentId, 2, 3),      
                'turvar_id' => (int)substr($datacontentId, 5, 1),   
                'tahun_id' => (int)substr($datacontentId, 6, 3),    
                'turtahun_id' => (int)substr($datacontentId, 9, 1), 
            ];
        } elseif ($length == 9) {
            // Format: vervar(1) + var(3) + turvar(1) + tahun(3) + turtahun(1) = 9 chars
            return [
                'vervar_id' => (int)substr($datacontentId, 0, 1),   
                'var_id' => (int)substr($datacontentId, 1, 3),      
                'turvar_id' => (int)substr($datacontentId, 4, 1),   
                'tahun_id' => (int)substr($datacontentId, 5, 3),    
                'turtahun_id' => (int)substr($datacontentId, 8, 1), 
            ];
        } elseif ($length >= 15) {
            // Original format: vervar(4) + var(3) + turvar(3) + tahun(3) + turtahun(2) = 15 chars
            return [
                'vervar_id' => (int)substr($datacontentId, 0, 4), 
                'var_id' => (int)substr($datacontentId, 4, 3),    
                'turvar_id' => (int)substr($datacontentId, 7, 3), 
                'tahun_id' => (int)substr($datacontentId, 10, 3), 
                'turtahun_id' => (int)substr($datacontentId, 13, 2), 
            ];
        }
        
        // Final fallback
        return [
            'vervar_id' => 0,
            'var_id' => 0,
            'turvar_id' => 0,
            'tahun_id' => 0,
            'turtahun_id' => 0,
        ];
    }

    private function calculateNilaiIndeks($datacontentVal, $parentData)
    {
        $jenisTangguh = $parentData['jenis_tangguh'];
        $datacontentVal = (float)$datacontentVal;
        
        // Define boundaries
        $boundaries = [
            ['min' => $parentData['batas_bawah'], 'max' => $parentData['batas_atas1']],
            ['min' => $parentData['batas_atas1'], 'max' => $parentData['batas_atas2']],
            ['min' => $parentData['batas_atas2'], 'max' => $parentData['batas_atas3']],
            ['min' => $parentData['batas_atas3'], 'max' => $parentData['batas_atas4']],
            ['min' => $parentData['batas_atas4'], 'max' => $parentData['batas_atas5']],
        ];

        // Determine which boundary the value falls into
        $indeks = 5; // default to highest category
        for ($i = 0; $i < count($boundaries); $i++) {
            if ($datacontentVal >= $boundaries[$i]['min'] && $datacontentVal < $boundaries[$i]['max']) {
                $indeks = $i + 1;
                break;
            }
            // Special case for the last boundary (includes upper limit)
            if ($i == count($boundaries) - 1 && $datacontentVal <= $boundaries[$i]['max']) {
                $indeks = $i + 1;
                break;
            }
        }

        // If jenis_tangguh is 'kebawah', invert the index
        // Semakin kecil nilai semakin baik, jadi nilai terkecil mendapat indeks tertinggi
        if ($jenisTangguh === 'kebawah') {
            $indeks = 6 - $indeks; // Convert 1->5, 2->4, 3->3, 4->2, 5->1
        }

        return $indeks;
    }

    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'gatra' => 'required|string|max:255',
            'type' => 'required|string|max:255',
            'turvar_id' => 'nullable|integer',
            'tahun_id' => 'nullable|integer',
            'turtahun_id' => 'nullable|integer',
            'jenis_tangguh' => 'required|string|max:255',
            'batas_bawah' => 'required|numeric',
            'batas_atas' => 'required|numeric',
            'api' => 'required|string|max:255',
        ]);

        $klasifikasi = Klasifikasi::findOrFail($id);
        $klasifikasi->update($validated);

        return redirect()->route('klasifikasi.index')->with('success', 'Data klasifikasi berhasil diperbarui.');
    }

    public function updateAllKlasifikasi()
    {
        $klasifikasiList = Klasifikasi::with('detailKlasifikasi')->get();
        
        foreach ($klasifikasiList as $klasifikasi) {
            // Recalculate nilai_indeks for all detail records
            foreach ($klasifikasi->detailKlasifikasi as $detail) {
                $parentData = [
                    'jenis_tangguh' => $klasifikasi->jenis_tangguh,
                    'batas_bawah' => $klasifikasi->batas_bawah,
                    'batas_atas1' => $klasifikasi->batas_atas1,
                    'batas_atas2' => $klasifikasi->batas_atas2,
                    'batas_atas3' => $klasifikasi->batas_atas3,
                    'batas_atas4' => $klasifikasi->batas_atas4,
                    'batas_atas5' => $klasifikasi->batas_atas5,
                ];

                $nilaiIndeks = $this->calculateNilaiIndeks($detail->datacontent_val, $parentData);
                $detail->update(['nilai_indeks' => $nilaiIndeks]);
            }
        }

        return redirect()->route('klasifikasi.index')->with('success', 'Semua data klasifikasi berhasil diperbarui.');
    }

    public function show($id)
    {
        $klasifikasi = Klasifikasi::with('detailKlasifikasi')->findOrFail($id);
        return response()->json([
            'klasifikasi' => $klasifikasi,
            'detailKlasifikasi' => $klasifikasi->detailKlasifikasi
        ]);
    }

    public function destroy($id)
    {
        $data_klasifikasi = Klasifikasi::findOrFail($id);
        
        // Delete all related detail_klasifikasi records first
        DetailKlasifikasi::where('klasifikasi_id', $id)->delete();
        
        // Then delete the klasifikasi record
        $data_klasifikasi->delete();

        return redirect()->route('klasifikasi.index')->with('success', 'Data klasifikasi dan detail klasifikasi berhasil dihapus.');
    }

    /**
     * API: Get Gatra and Klasifikasi data
     * Returns combined data from gatra and klasifikasi tables
     */
    public function getGatraKlasifikasiApi()
    {
        try {
            // Get all gatra data
            $gatraList = Gatra::all();
            
            // Get all klasifikasi data with selected fields joined with gatra
            $klasifikasiList = Klasifikasi::select(
                    'klasifikasi.id', 
                    'gatra.nama_gatra as gatra_name', 
                    'klasifikasi.subject', 
                    'klasifikasi.var'
                )
                ->leftJoin('gatra', 'klasifikasi.gatra', '=', 'gatra.id')
                ->orderBy('gatra.nama_gatra')
                ->orderBy('klasifikasi.subject')
                ->get();
            
            // Group klasifikasi by gatra name
            $klasifikasiByGatra = $klasifikasiList->groupBy('gatra_name');
            
            // Prepare response data
            $response = [];
            
            foreach ($gatraList as $gatra) {
                $gatraData = [
                    'gatra_id' => $gatra->id,
                    'gatra_name' => $gatra->nama_gatra,
                    'klasifikasi' => []
                ];
                
                // Find klasifikasi for this gatra
                $gatraName = $gatra->nama_gatra;
                if (isset($klasifikasiByGatra[$gatraName])) {
                    foreach ($klasifikasiByGatra[$gatraName] as $klasifikasi) {
                        $gatraData['klasifikasi'][] = [
                            'id' => $klasifikasi->id,
                            'gatra' => $klasifikasi->gatra_name,
                            'subject' => $klasifikasi->subject,
                            'var' => $klasifikasi->var
                        ];
                    }
                }
                
                $response[] = $gatraData;
            }
            
            return response()->json([
                'status' => 'success',
                'message' => 'Data berhasil diambil',
                'data' => $response
            ], 200);
            
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Terjadi kesalahan: ' . $e->getMessage(),
                'data' => []
            ], 500);
        }
    }

    /**
     * API: Get simplified Gatra-Klasifikasi data
     * Returns flat structure with id, gatra, subject, var
     */
    public function getGatraKlasifikasiSimpleApi()
    {
        try {
            // Get klasifikasi data joined with gatra information
            $data = Klasifikasi::select(
                    'klasifikasi.id', 
                    'gatra.gatra', 
                    'klasifikasi.subject', 
                    'klasifikasi.var'
                )
                ->leftJoin('gatra', 'klasifikasi.gatra', '=', 'gatra.id')
                ->orderBy('gatra.id')
                ->orderBy('klasifikasi.subject')
                ->get();
            
            $response = [];
            foreach ($data as $item) {
                $response[] = [
                    'id' => $item->id,
                    'gatra' => $item->gatra,
                    'subject' => $item->subject,
                    'var' => $item->var
                ];
            }
            
            return response()->json([
                'status' => 'success',
                'message' => 'Data berhasil diambil',
                'data' => $response
            ], 200);
            
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Terjadi kesalahan: ' . $e->getMessage(),
                'data' => []
            ], 500);
        }
    }
}