<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Models\Pelacakan;
use App\Models\Pelanggan;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PelacakanController extends Controller
{
    public function index(Request $request)
    {
        $no = 1;
        $title = 'Transaksi';

        // Ambil filter dari request
        $tanggal = $request->input('tanggal');
        $status = $request->input('jenis');

        // Mulai query untuk mengambil data pelacakan
        $query = Pelacakan::query();

        // Filter berdasarkan tanggal jika ada
        if ($tanggal) {
            $query->whereDate('created_at', $tanggal);
        }

        // Filter berdasarkan status jika ada
        if ($status) {
            $query->where('status', $status);
        }

        // Ambil semua data atau yang sudah difilter
        $data = $query->get();

        $orders_admin = $query->with(['customer', 'products', 'kurir'])->latest()->get();
        $orders_kurir = $query->with(['customer', 'products', 'kurir'])->where('status', '!=', 'dikemas')->latest()->get();
        $orders = auth()->user()->role == 'Kurir' ? $orders_kurir : $orders_admin;
        $kurir = User::where('role', 'Kurir')->get();

        return view('pages.pelacakan.index', compact('no', 'title', 'data', 'orders', 'kurir'));
    }


    public function create()
    {
        $title = 'Transaksi';
        $produk = Product::all();
        $customer = Pelanggan::all();
        return view('pages.pelacakan.create', compact('title', 'produk', 'customer'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'produk' => 'required',
            'jumlah_barang' => 'required|integer|min:1',
            'customer' => 'required',
        ]);

        $productId = $request->produk;
        $jumlahBarang = $request->jumlah_barang;
        $produk = Product::findOrFail($productId);
        $customer = Pelanggan::findOrFail($request->customer);

        if ($produk->stok_produk < $jumlahBarang) {
            return redirect()->back()->with('error', 'Stok tidak mencukupi untuk produk ' . $produk->nama_produk . ' !');
        }

        try {
            DB::transaction(function () use ($request, $produk, $jumlahBarang, $productId, $customer) {
                // Buat pelacakan
                $pelacakan = new Pelacakan();
                $pelacakan->id_karyawan = auth()->user()->id;
                $pelacakan->id_customer = $request->customer;
                $pelacakan->total = $request->total;
                $pelacakan->sisa_pelunasan = $request->total;
                $pelacakan->status = 'dikemas';
                $pelacakan->save();

                // Update stok produk
                $produk->decrement('stok_produk', $jumlahBarang);

                // Simpan data inventory
                $inventory = new Inventory();
                $inventory->jenis = 'barang keluar';
                $inventory->jumlah_barang = $jumlahBarang;
                $inventory->id_produk = $productId;
                $inventory->id_karyawan = auth()->user()->id;
                $inventory->pesan = "Pembelian " . $produk->nama_produk . " oleh " . $customer->nama_pelanggan;
                $inventory->pembayaran = $request->total;
                $inventory->save();

                // Update pelacakan dengan id_inventory
                $pelacakan->id_inventory = $inventory->id;
                $pelacakan->save();

                // Simpan relasi produk ke pelacakan
                $pelacakan->products()->attach($productId, [
                    'harga_produk' => $produk->harga_produk,
                    'quantity' => $jumlahBarang,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            });

            return redirect()->route('pelacakan.index')->with('success', 'Data berhasil disimpan!');
        } catch (\Exception $e) {
            return redirect()->route('pelacakan.index')->with('error', 'Gagal menyimpan data: ' . $e->getMessage());
        }
    }

    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:dikemas,dikirim,selesai,dibatalkan',
        ]);

        $pelacakan = Pelacakan::with('products', 'customer')->findOrFail($id);
        $oldStatus = $pelacakan->status;
        $newStatus = $request->status;

        try {
            DB::transaction(function () use ($pelacakan, $newStatus, $oldStatus) {
                $pelacakan->status = $newStatus;
                $pelacakan->id_karyawan = auth()->user()->id;
                $pelacakan->save();

                foreach ($pelacakan->products as $product) {
                    $quantity = $product->pivot->quantity;

                    if ($oldStatus !== 'dibatalkan' && $newStatus === 'dibatalkan') {
                        // Kembalikan stok produk
                        $product->increment('stok_produk', $quantity);

                        // Catat inventory: barang masuk
                        Inventory::create([
                            'jenis' => 'barang masuk',
                            'jumlah_barang' => $quantity,
                            'id_produk' => $product->id,
                            'id_karyawan' => auth()->user()->id,
                            'pesan' => "Pembatalan pesanan " . $product->nama_produk . " oleh " . $pelacakan->customer->nama_pelanggan,
                            'pembayaran' => 0,
                        ]);
                    }

                    if ($oldStatus === 'dibatalkan' && $newStatus !== 'dibatalkan') {
                        if ($product->stok_produk < $quantity) {
                            throw new \Exception("Stok produk '{$product->nama_produk}' tidak mencukupi.");
                        }

                        // Kurangi stok kembali
                        $product->decrement('stok_produk', $quantity);

                        // Catat inventory: barang keluar ulang
                        Inventory::create([
                            'jenis' => 'barang keluar',
                            'jumlah_barang' => $quantity,
                            'id_produk' => $product->id,
                            'id_karyawan' => auth()->user()->id,
                            'pesan' => "Pesanan aktif kembali untuk " . $product->nama_produk . " oleh " . $pelacakan->customer->nama_pelanggan,
                            'pembayaran' => 0,
                        ]);
                    }
                }
            });

            return redirect()->back()->with('success', 'Status berhasil diperbarui!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal mengupdate status: ' . $e->getMessage());
        }
    }

    public function updateKurir(Request $request, $id)
    {
        $request->validate([
            'kurir' => 'required|exists:users,id',
        ]);

        $user = User::findOrFail($request->kurir);
        if ($user->role != 'Kurir') {
            return redirect()->back()->with('error', 'User yang dipilih bukan kurir!');
        }

        $pelacakan = Pelacakan::findOrFail($id);
        $pelacakan->id_kurir = $request->kurir;
        $pelacakan->save();

        return redirect()->back()->with('success', 'Kurir berhasil diperbarui!');
    }

    public function productCancel($id)
    {
        try {
            DB::transaction(function () use ($id) {
                $pelacakan = Pelacakan::with('products', 'customer')->findOrFail($id);

                if ($pelacakan->status === 'dibatalkan') {
                    throw new \Exception('Pesanan sudah dibatalkan sebelumnya.');
                }

                // Update status
                $pelacakan->status = 'dibatalkan';
                $pelacakan->id_karyawan = auth()->user()->id;
                $pelacakan->save();

                foreach ($pelacakan->products as $product) {
                    // Tambahkan kembali stok
                    $product->increment('stok_produk', $product->pivot->quantity);

                    // Catat ke inventory
                    $inventory = new Inventory();
                    $inventory->jenis = 'barang masuk';
                    $inventory->jumlah_barang = $product->pivot->quantity;
                    $inventory->id_produk = $product->id;
                    $inventory->id_karyawan = auth()->user()->id;
                    $inventory->pesan = "Pembatalan pesanan " . $product->nama_produk . " oleh " . $pelacakan->customer->nama_pelanggan;
                    $inventory->pembayaran = 0; // Tidak ada uang masuk
                    $inventory->save();
                }
            });

            return redirect()->back()->with('success', 'Pesanan berhasil dibatalkan dan stok dikembalikan.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal membatalkan pesanan: ' . $e->getMessage());
        }
    }

    public function update($id, Request $request)
    {
        // dd($request);
        if ($request->file('bukti_pengiriman')) {
            $request->validate([
                'bukti_pengiriman' => 'required|image|mimes:jpeg,png,jpg|max:2048',
            ]);

            $buktiPath = $request->file('bukti_pengiriman')->store('bukti_pengiriman', 'public');
            $imageName = str_replace('bukti_pengiriman/', '', $buktiPath);

            $update = Pelacakan::findOrFail($id);
            $update->id_kurir = auth()->user()->id;
            $update->status = 'selesai';
            $update->bukti_pengiriman = $imageName;

            if ($update->save()) {
                return redirect()->back()->with('success', 'Status berhasil diperbarui!');
            } else {
                return redirect()->back()->with('error', 'Gagal mengupdate status');
            }
        } elseif ($request->has('pembayaran')) {
            $request->validate([
                'pembayaran' => 'required',
                'sisa' => 'required',
            ]);

            $update = Pelacakan::findOrFail($id);
            $update->jumlah_pelunasan = $request->pembayaran;
            $update->sisa_pelunasan = $request->sisa;

            if ($update->save()) {
                return redirect()->back()->with('success', 'Pembayaran berhasil diperbarui!');
            } else {
                return redirect()->back()->with('error', 'Gagal mengupdate pembayaran');
            }
        } else {
            $request->validate([
                'pembayaran_sisa' => 'required',
                'sisa_pembayaran' => 'required',
            ]);

            $update = Pelacakan::findOrFail($id);
            $pelunasan_sebelumnya = $update->jumlah_pelunasan;
            $update_pelunasan = $pelunasan_sebelumnya + $request->pembayaran_sisa;
            $update->jumlah_pelunasan = $update_pelunasan;
            $update->sisa_pelunasan = $request->sisa_pembayaran;

            if ($update->save()) {
                return redirect()->back()->with('success', 'Pembayaran berhasil diperbarui!');
            } else {
                return redirect()->back()->with('error', 'Gagal mengupdate pembayaran');
            }
        }
    }

    public function printLaporan(Request $request)
    {
        $request->validate([
            'from' => 'required|date',
            'to' => 'required|date|after_or_equal:from',
        ]);

        $from = $request->from;
        $to = $request->to;

        $pelacakans = Pelacakan::with(['customer', 'products', 'kurir'])
            ->whereBetween('created_at', [$from . ' 00:00:00', $to . ' 23:59:59'])
            ->orderBy('created_at', 'asc')
            ->get();

        return view('pages.pelacakan.print', compact('pelacakans', 'from', 'to'));
    }

    public function customerSelesai(Request $request, $id)
    {
        $request->validate([
            'bukti_penerimaan' => 'required|image|mimes:jpg,jpeg,png|max:2048',
        ]);

        $pelacakan = Pelacakan::findOrFail($id);

        if ($request->hasFile('bukti_penerimaan')) {
            $filename = time() . '_' . $request->file('bukti_penerimaan')->getClientOriginalName();
            $path = $request->file('bukti_penerimaan')->storeAs('public/bukti_pengiriman', $filename);
            $pelacakan->bukti_pengiriman = $filename;
        }

        $pelacakan->status = 'selesai';
        $pelacakan->save();

        return redirect()->back()->with('success', 'Pesanan telah diterima dan bukti berhasil dikirim.');
    }

    public function unduhPenjualanPerBulan(Request $request)
    {
        $tahun = $request->input('tahun');
        $bulan = $request->input('bulan');

        if (!$tahun || !$bulan) {
            return redirect()->back()->with('error', 'Tahun dan bulan harus dipilih.');
        }

        $orders = Pelacakan::with(['products', 'customer', 'staff'])
            ->whereYear('created_at', $tahun)
            ->whereMonth('created_at', $bulan)
            ->get();

        $kurir = []; // jika perlu
        $no = 1;

        return view('pages.pelacakan.unduh_per_bulan', [
            'orders' => $orders,
            'kurir' => $kurir,
            'no' => $no,
            'selectedBulan' => $bulan,
            'selectedTahun' => $tahun,
            'nama_bulan' => Carbon::create()->month($bulan)->translatedFormat('F'),
        ]);
    }
}
