<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Models\Pelacakan;
use App\Models\Product;
use App\Models\Sales;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ProductController extends Controller
{
    public function index()
    {
        $no = 1;
        $data_product = Product::latest()->get();
        $sales = Sales::orderBy('nama')->get();
        return view('pages.data-product.index', compact('no', 'data_product', 'sales'));
    }

    public function create()
    {
        return view('pages.data-product.create'); // Arahkan ke view form create
    }

    // Metode untuk menyimpan data produk baru
    public function store(Request $request)
    {
        // Debug: log semua input request
        \Log::info('=== PRODUCT STORE DEBUG START ===');
        \Log::info('All request data:', $request->all());
        \Log::info('Has file foto_produk: ' . ($request->hasFile('foto_produk') ? 'YES' : 'NO'));
        \Log::info('Request method: ' . $request->method());
        \Log::info('Request URL: ' . $request->url());

        // Coba validasi step by step
        try {
            $request->validate([
                'nama_produk' => 'required|string|max:255',
                'foto_produk' => 'nullable|file|mimes:jpeg,png,jpg,gif,webp|max:2048',
                'nama_sales' => 'required|exists:sales,id',
                'harga_produk' => 'required|numeric|min:0',
                'stok_produk' => 'required|integer|min:0',
                'best_seller' => 'required|integer|min:0|max:1',
            ], [
                'nama_produk.required' => 'Nama produk wajib diisi',
                'foto_produk.file' => 'File harus berupa file yang valid',
                'foto_produk.mimes' => 'Format gambar harus jpeg, png, jpg, gif, atau webp',
                'foto_produk.max' => 'Ukuran gambar maksimal 2MB',
                'nama_sales.required' => 'Sales wajib dipilih',
                'nama_sales.exists' => 'Sales tidak valid',
                'harga_produk.required' => 'Harga produk wajib diisi',
                'harga_produk.numeric' => 'Harga produk harus berupa angka',
                'harga_produk.min' => 'Harga produk tidak boleh minus',
                'stok_produk.required' => 'Stok produk wajib diisi',
                'stok_produk.integer' => 'Stok produk harus berupa angka bulat',
                'stok_produk.min' => 'Stok produk tidak boleh minus',
                'best_seller.required' => 'Status best seller wajib dipilih',
            ]);
            
            \Log::info('Validation passed successfully');
        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation failed:', $e->errors());
            throw $e;
        }

        // Default image name
        $imagePath = 'default-product.svg';

        // Handle image upload
        if ($request->hasFile('foto_produk')) {
            try {
                \Log::info('Processing file upload...');
                $file = $request->file('foto_produk');
                \Log::info('File info:', [
                    'original_name' => $file->getClientOriginalName(),
                    'size' => $file->getSize(),
                    'mime_type' => $file->getMimeType()
                ]);
                
                $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->storeAs('foto_produk', $fileName, 'public');
                $imagePath = $fileName;
                \Log::info('File uploaded successfully as: ' . $fileName);
            } catch (\Exception $e) {
                \Log::error('File upload error: ' . $e->getMessage());
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['foto_produk' => 'Gagal mengupload gambar: ' . $e->getMessage()]);
            }
        } else {
            \Log::info('No file uploaded, using default image');
        }

        try {
            // Debug: log data sebelum disimpan
            \Log::info('Data yang akan disimpan:', [
                'nama_produk' => $request->nama_produk,
                'foto_produk' => $imagePath,
                'harga_produk' => $request->harga_produk,
                'stok_produk' => $request->stok_produk,
                'best_seller' => $request->best_seller,
                'id_sales' => $request->nama_sales,
            ]);

            $product = Product::create([
                'nama_produk' => $request->nama_produk,
                'foto_produk' => $imagePath,
                'harga_produk' => $request->harga_produk,
                'stok_produk' => $request->stok_produk,
                'best_seller' => $request->best_seller,
                'id_sales' => $request->nama_sales,
            ]);

            \Log::info('Produk berhasil disimpan dengan ID: ' . $product->id);

            return redirect()->route('data-product.index')->with('success', 'Produk berhasil ditambahkan dengan ID: ' . $product->id);
        } catch (\Exception $e) {
            \Log::error('Error menyimpan produk: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Gagal menyimpan produk: ' . $e->getMessage()]);
        }
    }

    public function edit($id)
    {
        $data_product = Product::findOrFail($id);
        return view('pages.data-product.edit', compact('product'));
    }

    // Memperbarui data produk
    public function update(Request $request, $id)
    {
        $request->validate([
            'nama_produk' => 'required|string|max:255',
            'foto_produk' => 'nullable|file|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'nama_sales' => 'required|exists:sales,id',
            'harga_produk' => 'required|numeric|min:0',
            'stok_produk' => 'required|integer|min:0',
            'best_seller' => 'required|integer|min:0|max:1',
        ], [
            'nama_produk.required' => 'Nama produk wajib diisi',
            'foto_produk.file' => 'File harus berupa file yang valid',
            'foto_produk.mimes' => 'Format gambar harus jpeg, png, jpg, gif, atau webp',
            'foto_produk.max' => 'Ukuran gambar maksimal 2MB',
            'nama_sales.required' => 'Sales wajib dipilih',
            'nama_sales.exists' => 'Sales tidak valid',
            'harga_produk.required' => 'Harga produk wajib diisi',
            'harga_produk.numeric' => 'Harga produk harus berupa angka',
            'harga_produk.min' => 'Harga produk tidak boleh minus',
            'stok_produk.required' => 'Stok produk wajib diisi',
            'stok_produk.integer' => 'Stok produk harus berupa angka bulat',
            'stok_produk.min' => 'Stok produk tidak boleh minus',
            'best_seller.required' => 'Status best seller wajib dipilih',
        ]);

        $product = Product::findOrFail($id);

        $data = [
            'nama_produk' => $request->nama_produk,
            'harga_produk' => $request->harga_produk,
            'stok_produk' => $request->stok_produk,
            'best_seller' => $request->best_seller,
            'id_sales' => $request->nama_sales,
        ];

        // Handle image upload
        if ($request->hasFile('foto_produk')) {
            try {
                // Delete old image if exists (except default image)
                if ($product->foto_produk && $product->foto_produk !== 'default-product.svg') {
                    Storage::disk('public')->delete('foto_produk/' . $product->foto_produk);
                }

                $file = $request->file('foto_produk');
                $fileName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                $file->storeAs('foto_produk', $fileName, 'public');
                $data['foto_produk'] = $fileName;
            } catch (\Exception $e) {
                return redirect()->back()
                    ->withInput()
                    ->withErrors(['foto_produk' => 'Gagal mengupload gambar: ' . $e->getMessage()]);
            }
        }

        try {
            $product->update($data);
            return redirect()->route('data-product.index')->with('success', 'Produk berhasil diperbarui.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Gagal memperbarui produk: ' . $e->getMessage()]);
        }
    }

    // Menghapus produk
    public function destroy($id)
    {
        $product = Product::findOrFail($id);

        if ($product->delete()) {
            if ($product->foto_produk) {
                // Delete product image in storage
                Storage::disk('public')->delete('foto_produk/' . $product->foto_produk);
            }
            return redirect()->back()->with('success', 'Data berhasil dihapus!');
        } else {
            return redirect()->back()->with('error', 'Gagal menghapus data');
        }
    }
}
